// ELOXP Chrome Extension - Popup Script

document.addEventListener('DOMContentLoaded', init);

// ============================================
// State
// ============================================

let currentUser = null;
let currentStats = null;

// ============================================
// Views
// ============================================

function showView(viewId) {
  // Hide ALL views first
  const views = document.querySelectorAll('.view');
  views.forEach(v => {
    v.classList.add('hidden');
    v.style.display = 'none';
  });
  
  // Show the requested view
  const targetView = document.getElementById(viewId);
  if (targetView) {
    targetView.classList.remove('hidden');
    targetView.style.display = '';
  }
}

// ============================================
// Auth Functions
// ============================================

async function checkAuth() {
  showView('loading-view');
  
  const response = await chrome.runtime.sendMessage({ type: 'CHECK_AUTH' });
  
  if (response.authenticated && response.user) {
    currentUser = response.user;
    await loadStats();
    showMainView();
  } else {
    showView('login-view');
  }
}

async function login() {
  const token = document.getElementById('token-input').value.trim();
  const errorEl = document.getElementById('login-error');
  const loginBtn = document.getElementById('login-btn');
  
  if (!token) {
    errorEl.textContent = 'Please enter your extension token';
    return;
  }
  
  loginBtn.textContent = 'Connecting...';
  loginBtn.disabled = true;
  errorEl.textContent = '';
  
  const response = await chrome.runtime.sendMessage({ 
    type: 'LOGIN', 
    payload: { token } 
  });
  
  if (response.success && response.user) {
    currentUser = response.user;
    await loadStats();
    showMainView();
  } else {
    errorEl.textContent = response.error || 'Invalid token. Please try again.';
    loginBtn.textContent = 'Connect';
    loginBtn.disabled = false;
  }
}

async function logout() {
  await chrome.runtime.sendMessage({ type: 'LOGOUT' });
  currentUser = null;
  currentStats = null;
  document.getElementById('token-input').value = '';
  showView('login-view');
}

// ============================================
// Stats Functions
// ============================================

async function loadStats() {
  const response = await chrome.runtime.sendMessage({ type: 'GET_STATS' });
  
  if (response.error) {
    console.error('Failed to load stats:', response.error);
    return;
  }
  
  currentStats = response;
  updateUI();
}

// ============================================
// UI Updates
// ============================================

function showMainView() {
  // Use the standard showView to hide ALL views and show main
  showView('main-view');
  updateUI();
}

function updateUI() {
  if (!currentUser && !currentStats) return;
  
  const user = currentStats?.user || currentUser;
  const rating = currentStats?.rating || {};
  const eloxp = currentStats?.eloxp || {};
  const extStats = currentStats?.extensionStats || {};
  const platforms = currentStats?.platforms || {};
  const tier = currentStats?.tier || {};
  
  // User info
  document.getElementById('user-name').textContent = user?.displayName || 'Player';
  
  // Avatar
  const avatarEl = document.getElementById('user-avatar');
  if (user?.avatarUrl) {
    avatarEl.innerHTML = `<img src="${user.avatarUrl}" alt="Avatar">`;
  }
  
  // Division
  const divisionEl = document.getElementById('user-division');
  const division = rating.division || currentUser?.division || 'rookie';
  divisionEl.textContent = division.charAt(0).toUpperCase() + division.slice(1);
  divisionEl.className = `division-badge ${division}`;
  
  // ELO
  document.getElementById('user-elo').textContent = 
    `${rating.current || currentUser?.eloRating || 800} ELO`;
  
  // Tier
  const tierEl = document.getElementById('user-tier');
  const tierName = (tier.current || currentUser?.subscriptionTier || 'free').toUpperCase();
  tierEl.textContent = tierName;
  tierEl.className = `user-tier ${tierName.toLowerCase()}`;
  
  // Stats
  document.getElementById('stat-lifetime').textContent = 
    formatNumber(eloxp.lifetime || currentUser?.lifetimeEloxp || 0);
  document.getElementById('stat-today').textContent = 
    formatNumber(eloxp.todaySession || 0);
  document.getElementById('stat-games').textContent = 
    eloxp.todayGamesPlayed || 0;
  document.getElementById('stat-winrate').textContent = 
    `${extStats.winRate || 0}%`;
  
  // Progress
  if (rating.current && rating.minElo !== undefined && rating.maxElo) {
    const progress = rating.progressToNext || 0;
    document.getElementById('progress-text').textContent = 
      `${rating.current} / ${rating.maxElo}`;
    document.getElementById('progress-fill').style.width = `${progress}%`;
    document.getElementById('progress-next').textContent = 
      rating.nextDivision ? `Next: ${capitalize(rating.nextDivision)}` : 'Max division!';
  }
  
  // Platforms
  updatePlatform('lichess', platforms.lichess);
  updatePlatform('chesscom', platforms.chesscom);
  
  // Recent games
  updateRecentGames(currentStats?.recentGames || []);
  
  // Settings
  loadSettings();
}

function updatePlatform(platform, data) {
  const el = document.getElementById(`platform-${platform}`);
  const statusEl = el.querySelector('.platform-status');
  
  if (data?.username) {
    statusEl.textContent = data.username;
    statusEl.classList.add('connected');
  } else {
    statusEl.textContent = 'Not connected';
    statusEl.classList.remove('connected');
  }
}

function updateRecentGames(games) {
  const listEl = document.getElementById('games-list');
  
  if (!games || games.length === 0) {
    listEl.innerHTML = '<div class="no-games">No games tracked yet</div>';
    return;
  }
  
  listEl.innerHTML = games.slice(0, 5).map(game => `
    <div class="game-item">
      <div class="game-result ${game.result}">
        ${game.result === 'win' ? 'W' : game.result === 'loss' ? 'L' : 'D'}
      </div>
      <div class="game-info">
        <div class="game-opponent">vs ${game.opponent || 'Unknown'}</div>
        <div class="game-platform">${capitalize(game.platform)} • ${formatTime(game.completedAt)}</div>
      </div>
      <div class="game-eloxp">+${game.eloxpEarned}</div>
    </div>
  `).join('');
}

async function loadSettings() {
  const settings = await chrome.runtime.sendMessage({ type: 'GET_SETTINGS' });
  document.getElementById('hud-toggle').checked = settings?.hudEnabled !== false;
}

async function toggleHUD() {
  const enabled = document.getElementById('hud-toggle').checked;
  const settings = await chrome.runtime.sendMessage({ type: 'GET_SETTINGS' });
  await chrome.runtime.sendMessage({ 
    type: 'UPDATE_SETTINGS', 
    payload: { ...settings, hudEnabled: enabled } 
  });
}

// ============================================
// Utilities
// ============================================

function formatNumber(num) {
  // Display full numbers with commas for clarity
  return num.toLocaleString();
}

function capitalize(str) {
  if (!str) return '';
  return str.charAt(0).toUpperCase() + str.slice(1);
}

function formatTime(dateStr) {
  if (!dateStr) return '';
  const date = new Date(dateStr);
  const now = new Date();
  const diff = now - date;
  
  if (diff < 60000) return 'Just now';
  if (diff < 3600000) return `${Math.floor(diff / 60000)}m ago`;
  if (diff < 86400000) return `${Math.floor(diff / 3600000)}h ago`;
  return date.toLocaleDateString();
}

// ============================================
// Event Listeners
// ============================================

function setupEventListeners() {
  document.getElementById('login-btn').addEventListener('click', login);
  document.getElementById('logout-btn').addEventListener('click', logout);
  document.getElementById('hud-toggle').addEventListener('change', toggleHUD);
  
  // Allow Enter key to submit token
  document.getElementById('token-input').addEventListener('keypress', (e) => {
    if (e.key === 'Enter') login();
  });
}

// ============================================
// Init
// ============================================

async function init() {
  setupEventListeners();
  await checkAuth();
}
