// ELOXP Chrome Extension - Background Service Worker
// Handles API communication, state management, and message routing

const API_BASE = 'https://eloxp.com';

// ============================================
// Storage Helpers
// ============================================

async function getStoredData(key) {
  const result = await chrome.storage.local.get(key);
  return result[key];
}

async function setStoredData(key, value) {
  await chrome.storage.local.set({ [key]: value });
}

async function getToken() {
  return await getStoredData('eloxp_token');
}

async function getUser() {
  return await getStoredData('eloxp_user');
}

async function getSettings() {
  const settings = await getStoredData('eloxp_settings');
  return settings || {
    hudEnabled: true,
    hudPosition: 'bottom-right',
    showNotifications: true
  };
}

// ============================================
// API Helpers
// ============================================

async function apiRequest(endpoint, options = {}) {
  const token = await getToken();
  
  const headers = {
    'Content-Type': 'application/json',
    ...(token && { 'Authorization': `Bearer ${token}` })
  };

  try {
    const response = await fetch(`${API_BASE}${endpoint}`, {
      ...options,
      headers: {
        ...headers,
        ...options.headers
      }
    });

    const data = await response.json();

    if (!response.ok) {
      if (response.status === 401) {
        // Token expired or invalid
        await setStoredData('eloxp_token', null);
        await setStoredData('eloxp_user', null);
        return { error: data.error || 'Session expired. Please log in again.', status: 401 };
      }
      return { error: data.error || 'API request failed', status: response.status };
    }

    return data;
  } catch (error) {
    console.error('ELOXP API Error:', error);
    return { error: 'Network error. Please check your connection.', offline: true };
  }
}

// ============================================
// Auth Functions
// ============================================

async function validateToken(token) {
  if (!token) return { error: 'No token provided' };

  const result = await apiRequest('/api/extension/auth', {
    method: 'GET',
    headers: { 'Authorization': `Bearer ${token}` }
  });

  if (result.valid && result.user) {
    await setStoredData('eloxp_token', token);
    await setStoredData('eloxp_user', result.user);
    return { success: true, user: result.user };
  }

  return { error: result.error || 'Invalid token' };
}

async function logout() {
  await setStoredData('eloxp_token', null);
  await setStoredData('eloxp_user', null);
  return { success: true };
}

// ============================================
// Game Tracking Functions
// ============================================

let currentGame = null;

async function trackGame(gameData) {
  const result = await apiRequest('/api/extension/track-game', {
    method: 'POST',
    body: JSON.stringify(gameData)
  });

  if (result.success) {
    currentGame = {
      ...gameData,
      internalId: result.gameId,
      eloxp: result.currentEloxp
    };
  }

  return result;
}

async function getGameStatus(platform, gameId) {
  return await apiRequest(`/api/extension/track-game?platform=${platform}&gameId=${gameId}`);
}

async function completeGame(gameData) {
  const result = await apiRequest('/api/extension/complete-game', {
    method: 'POST',
    body: JSON.stringify(gameData)
  });

  if (result.success) {
    currentGame = null;
    // Update cached user data with new ELOXP
    if (result.profile) {
      const user = await getUser();
      if (user) {
        user.lifetimeEloxp = result.profile.totalLifetimeEloxp;
        await setStoredData('eloxp_user', user);
      }
    }
  }

  return result;
}

// ============================================
// Stats Functions
// ============================================

async function getStats() {
  return await apiRequest('/api/extension/stats');
}

// ============================================
// Message Handler
// ============================================

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  handleMessage(message, sender).then(sendResponse);
  return true; // Keep channel open for async response
});

async function handleMessage(message, sender) {
  const { type, payload } = message;

  switch (type) {
    // Auth messages
    case 'LOGIN':
      return await validateToken(payload.token);

    case 'LOGOUT':
      return await logout();

    case 'CHECK_AUTH':
      const token = await getToken();
      const user = await getUser();
      return { authenticated: !!token && !!user, user };

    // Stats messages
    case 'GET_STATS':
      return await getStats();

    case 'GET_USER':
      return { user: await getUser() };

    // Game tracking messages (from content scripts)
    case 'GAME_STARTED':
      console.log('Game started:', payload);
      return await trackGame(payload);

    case 'MOVE_MADE':
      console.log('Move made:', payload.moveNumber);
      return await trackGame(payload);

    case 'GAME_ENDED':
      console.log('Game ended:', payload);
      return await completeGame(payload);

    case 'GET_GAME_STATUS':
      return await getGameStatus(payload.platform, payload.gameId);

    // Settings messages
    case 'GET_SETTINGS':
      return await getSettings();

    case 'UPDATE_SETTINGS':
      await setStoredData('eloxp_settings', payload);
      // Notify content scripts of settings change
      chrome.tabs.query({}, (tabs) => {
        tabs.forEach(tab => {
          if (tab.url?.includes('lichess.org') || tab.url?.includes('chess.com')) {
            chrome.tabs.sendMessage(tab.id, { type: 'SETTINGS_UPDATED', payload });
          }
        });
      });
      return { success: true };

    // Opponent lookup - check if opponent is an ELOXP fighter
    case 'LOOKUP_OPPONENT':
      return await lookupOpponent(payload.platform, payload.username);

    default:
      console.log('Unknown message type:', type);
      return { error: 'Unknown message type' };
  }
}

// ============================================
// Opponent Lookup - Check if opponent is on ELOXP
// ============================================

async function lookupOpponent(platform, username) {
  return await apiRequest(`/api/extension/lookup-opponent?platform=${platform}&username=${encodeURIComponent(username)}`);
}

// ============================================
// Startup
// ============================================

chrome.runtime.onInstalled.addListener(() => {
  console.log('ELOXP Extension installed!');
  // Initialize default settings
  getSettings().then(settings => {
    setStoredData('eloxp_settings', settings);
  });
});

// Validate token on startup
chrome.runtime.onStartup.addListener(async () => {
  const token = await getToken();
  if (token) {
    const result = await validateToken(token);
    if (result.error) {
      console.log('Token validation failed:', result.error);
    }
  }
});

// ============================================
// Keyboard Shortcut Handler (Ctrl+Shift+E)
// ============================================

chrome.commands.onCommand.addListener(async (command) => {
  if (command === 'toggle-hud') {
    console.log('ELOXP: Toggle HUD command received');
    
    // Get current tab
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    if (tab && (tab.url?.includes('lichess.org') || tab.url?.includes('chess.com'))) {
      // Send toggle message to content script
      chrome.tabs.sendMessage(tab.id, { type: 'TOGGLE_HUD' });
    }
  }
});
